﻿using System.Collections.Generic;

namespace Marimo.開発支援.コンソールアプリケーションライブラリ
{
    /// <summary>
    /// コマンドライン引数の実際の操作を実現するクラスです。
    /// </summary>
    internal class コマンドライン引数操作
    {
        /// <summary>
        /// スラッシュの文字列「/」です。
        /// </summary>
        private const string スラッシュ = "/";

        /// <summary>
        /// ハイフンの文字列「-」です。
        /// </summary>
        private const string ハイフン = "-";

        /// <summary>
        /// 大文字と小文字の区別を現在の設定どおりにして、エイリアスとコマンドを関連付けた辞書です。区別しない場合は小文字にしたエイリアスと関連付けられています。
        /// </summary>
        private readonly Dictionary<string, コマンド> エイリアス辞書 = new Dictionary<string, コマンド>();

        /// <summary>
        /// エイリアスの大文字と小文字を区別した状態で、エイリアスとコマンドを関連付けた辞書です。
        /// </summary>
        private readonly Dictionary<string, コマンド> 大文字小文字を区別したエイリアス辞書 = new Dictionary<string, コマンド>();

        /// <summary>
        /// 登録してあるコマンドです。
        /// </summary>
        private readonly List<コマンド> 登録コマンドリスト = new List<コマンド>();

        /// <summary>
        /// 大文字と小文字を区別することを示す値です。
        /// </summary>
        private bool 大文字小文字を区別するフィールド;

        /// <summary>
        /// 大文字と小文字を区別することを示す値を取得、設定します。
        /// </summary>
        internal bool 大文字小文字を区別する
        {
            get { return 大文字小文字を区別するフィールド; }
            set
            {
                大文字小文字を区別するフィールド = value;
                エイリアスを再登録する();
            }
        }

        /// <summary>
        /// 大文字と小文字を区別したエイリアス登録を元に、大文字と小文字の区別などの設定に合わせてエイリアスとコマンドを関連付ける辞書を再登録します。
        /// </summary>
        private void エイリアスを再登録する()
        {
            エイリアス辞書.Clear();

            foreach (var 登録するエイリアス in 大文字小文字を区別したエイリアス辞書)
            {
                エイリアスが登録されているかチェックする(大文字小文字を区別しないなら小文字に変換する(登録するエイリアス.Key));

                エイリアス辞書.Add(大文字小文字を区別しないなら小文字に変換する(登録するエイリアス.Key), 登録するエイリアス.Value);
            }
        }

        /// <summary>
        /// エイリアスに対応付けられたコマンドを取得します。
        /// </summary>
        /// <param name="エイリアス">コマンドと対応するエイリアス。</param>
        /// <returns>エイリアスと対応するコマンド。</returns>
        private コマンド コマンドを取得する(string エイリアス)
        {
            return エイリアス辞書[大文字小文字を区別しないなら小文字に変換する(エイリアス)];
        }

        /// <summary>
        /// コマンドの登録に、コマンドを識別するキーとして登録されています。
        /// </summary>
        /// <param name="コマンドキー">コマンドを識別するキー。</param>
        /// <returns></returns>
        private bool コマンドキーとして登録されている(string コマンドキー)
        {
            return 登録コマンドリスト.Exists(登録コマンド =>
                                    大文字小文字を区別しないなら小文字に変換する(登録コマンド.コマンドキー) ==
                                    大文字小文字を区別しないなら小文字に変換する(コマンドキー));
        }

        /// <summary>
        /// エイリアスとして登録されているかどうかを示す値を取得します。
        /// </summary>
        /// <param name="エイリアス">判別するエイリアス。</param>
        /// <returns>エイリアスとして登録されているかどうかを示す値。</returns>
        private bool エイリアスとして登録されている(string エイリアス)
        {
            return エイリアス辞書.ContainsKey(大文字小文字を区別しないなら小文字に変換する(エイリアス));
        }

        /// <summary>
        /// エイリアスとコマンドの対応をを登録します。
        /// </summary>
        /// <param name="エイリアス">登録するエイリアス。</param>
        /// <param name="コマンド">エイリアスに対応付けられたコマンド。</param>
        private void エイリアスを登録する(string エイリアス, コマンド コマンド)
        {
            エイリアスが登録されているかチェックする(エイリアス);

            エイリアス辞書.Add(大文字小文字を区別しないなら小文字に変換する(エイリアス), コマンド);
            大文字小文字を区別したエイリアス辞書.Add(エイリアス, コマンド);
        }

        /// <summary>
        /// エイリアスが、エイリアスもしくはコマンドキー登録してあるかどうかをチェックし、問題があれば例外を発生させます。
        /// </summary>
        /// <param name="エイリアス"></param>
        private void エイリアスが登録されているかチェックする(string エイリアス)
        {
            if (エイリアスとして登録されている(エイリアス))
            {
                throw new 例外管理.エイリアス登録重複(大文字小文字を区別しないなら小文字に変換する(エイリアス));
            }

            if (コマンドキーとして登録されている(エイリアス))
            {
                throw new 例外管理.コマンドキーと重複するエイリアス登録(大文字小文字を区別しないなら小文字に変換する(エイリアス));
            }
        }

        /// <summary>
        /// コマンドキーが、エイリアスもしくはコマンドキー登録してあるかどうかをチェックし、問題があれば例外を発生させます。
        /// </summary>
        /// <param name="コマンドキー"></param>
        private void コマンドキーが登録されているかチェックする(string コマンドキー)
        {
            if (コマンドキーとして登録されている(コマンドキー))
            {
                throw new 例外管理.コマンド登録重複(コマンドキー);
            }

            if (エイリアスとして登録されている(コマンドキー))
            {
                throw new 例外管理.コマンドキーと重複するエイリアス登録(大文字小文字を区別しないなら小文字に変換する(コマンドキー));
            }
        }

        /// <summary>
        /// コマンドを登録します。
        /// </summary>
        /// <param name="コマンド"></param>
        internal void コマンドを登録する(コマンド コマンド)
        {
            コマンドキーが登録されているかチェックする(コマンド.コマンドキー);

            登録コマンドリスト.Add(コマンド);

            foreach (string エイリアス in コマンド.エイリアスリスト)
            {
                エイリアスを登録する(エイリアス, コマンド);
            }
        }

        /// <summary>
        /// コマンドのキーとして認識されることを表す値を取得します。
        /// </summary>
        /// <param name="arg">コマンドかどうかを判別するコマンドライン引数文字列。</param>
        /// <returns>コマンドであることを表す値。</returns>
        private static bool コマンドである(string arg)
        {
            return arg.StartsWith(ハイフン) || arg.StartsWith(スラッシュ);
        }

        /// <summary>
        /// 登録されたコマンドを検索し、コマンドライン文字列の配列からコマンドの情報を読み取ります。
        /// </summary>
        /// <param name="args">コマンドライン引数の配列。</param>
        /// <returns>コマンドの情報。</returns>
        internal Dictionary<string, string> コマンド一覧を取得する(string[] args)
        {
            var 戻り値 = new Dictionary<string, string>();

            for (int i = 0; i < args.Length; i++)
            {
                if (!コマンドである(args[i]))
                {
                    throw new 例外管理.コマンド引数不正(args);
                }

                string コマンドラインで指定されたコマンド;
                string 指定されたコマンドのキー;
                string 指定されたコマンドの引数;

                引数から関連文字列を取得する(args, i, out コマンドラインで指定されたコマンド, out 指定されたコマンドのキー, out 指定されたコマンドの引数);

                if (指定されたコマンドの引数 != null)
                {
                    i++;
                }

                if (!コマンドキーとして登録されている(指定されたコマンドのキー) && !エイリアスとして登録されている(指定されたコマンドのキー))
                {
                    throw new 例外管理.未登録コマンド呼び出し(コマンドラインで指定されたコマンド);
                }

                if (戻り値.ContainsKey(登録されたキーに変換する(指定されたコマンドのキー)))
                {
                    throw new 例外管理.コマンド指定重複(登録されたキーに変換する(指定されたコマンドのキー));
                }

                戻り値.Add(登録されたキーに変換する(指定されたコマンドのキー), 指定されたコマンドの引数);
            }

            必須コマンドがすべて指定されているかチェックする(戻り値);

            return 戻り値;
        }

        /// <summary>
        /// 引数の列から読み込むコマンドの位置を指定し、コマンドに関連する文字列であるコマンドラインで指定されたコマンド、指定されたコマンドのキー、指定されたコマンドの引数の三つを取得します。
        /// </summary>
        /// <param name="args">コマンドライン引数の配列。</param>
        /// <param name="コマンド位置">今回の取得対象となるコマンドの位置。</param>
        /// <param name="コマンドラインで指定されたコマンド">今回実際に取得対象となるコマンドライン引数の文字列。</param>
        /// <param name="指定されたコマンドのキー">ハイフンやスラッシュを含まないコマンドキー。</param>
        /// <param name="指定されたコマンドの引数">コマンドに対応する引数。</param>
        private void 引数から関連文字列を取得する(string[] args, int コマンド位置, out string コマンドラインで指定されたコマンド, out string 指定されたコマンドのキー,
                                    out string 指定されたコマンドの引数)
        {
            コマンドラインで指定されたコマンド = args[コマンド位置];
            指定されたコマンドのキー = コマンドラインで指定されたコマンド.Substring(1);

            if (コマンド位置 + 1 < args.Length && !コマンドである(args[コマンド位置 + 1]))
            {
                指定されたコマンドの引数 = args[コマンド位置 + 1];
            }
            else
            {
                指定されたコマンドの引数 = null;
            }
        }

        /// <summary>
        /// コマンドキーを、大文字小文字の区別などを考慮して、実際に登録されているコマンドキーに変換します。
        /// </summary>
        /// <param name="入力されたコマンドキー"></param>
        /// <returns></returns>
        private string 登録されたキーに変換する(string 入力されたコマンドキー)
        {
            if (コマンドキーとして登録されている(入力されたコマンドキー))
            {
                return 登録コマンドリスト.Find(登録コマンド =>
                                      大文字小文字を区別しないなら小文字に変換する(登録コマンド.コマンドキー) ==
                                      大文字小文字を区別しないなら小文字に変換する(入力されたコマンドキー)
                    ).コマンドキー;
            }
            return コマンドを取得する(大文字小文字を区別しないなら小文字に変換する(入力されたコマンドキー)).コマンドキー;
        }

        /// <summary>
        /// 大文字小文字を区別しない状態の時には文字列を小文字に変換し、そうでない場合はそのままのものを取得します。
        /// </summary>
        /// <param name="変換前">変換対象となる文字列。</param>
        /// <returns>必要に応じて変換した文字列。</returns>
        private string 大文字小文字を区別しないなら小文字に変換する(string 変換前)
        {
            if (大文字小文字を区別する)
            {
                return 変換前;
            }
            else
            {
                return 変換前.ToLower();
            }
        }

        /// <summary>
        /// すべてのコマンドをチェックし、必須であるのに指定されていないコマンドがあれば例外を発生させます。
        /// </summary>
        /// <param name="指定されたコマンド一覧">チェック対象となるコマンド一覧。</param>
        private void 必須コマンドがすべて指定されているかチェックする(Dictionary<string, string> 指定されたコマンド一覧)
        {
            foreach (コマンド コマンド in 登録コマンドリスト)
            {
                if (コマンド.必須である && !指定されたコマンド一覧.ContainsKey(登録されたキーに変換する(コマンド.コマンドキー)))
                {
                    throw new 例外管理.必須コマンド不足(コマンド);
                }
            }
        }
    }
}